package editor;

import game.CustomImageDataII;
import game.IntRect;
import game.RainbowElementalSprite;

import java.awt.Color;
import java.awt.Graphics;
import java.awt.Point;
import java.awt.image.ImageObserver;
import java.io.BufferedWriter;

public class RainbowBossScreenObject extends ScreenObject {
	
	private RainbowIntRect clickRect;
	
	private CustomImageDataII noSpawnImage;
	private CustomImageDataII spawnTwoImage;
	private CustomImageDataII spawnSixImage;
	private CustomImageDataII cloudImage;
	private CustomImageDataII flagOutlineImage;
	
	private int type;
	
	private boolean isSelected = false;
	
	private Point flagSpawnPoint;
	
	private static final int ADJUSTED_RADIUS = RainbowElementalSprite.MOVEMENT_RADIUS * 9 / 10;
	
	private RainbowBossScreenPaletteObject paletteObject;
	
	
	public RainbowBossScreenObject(int centerX, int centerY, ImageObserver io, int theType, RainbowBossScreenPaletteObject thePaletteObject) {
		
		type = theType;
		paletteObject = thePaletteObject;
		
		cloudImage = new CustomImageDataII("/images/boss/cloud spawn A 3.GIF", Color.white, io);
		noSpawnImage = new CustomImageDataII("/images/boss/small_rainbow.GIF", Color.black, io);
		spawnTwoImage = new CustomImageDataII("/images/boss/med_rainbow.GIF", Color.black, io);
		spawnSixImage = new CustomImageDataII("/images/boss/rainbow2.GIF", Color.black, io);
		flagOutlineImage = new CustomImageDataII("/images/flagoutline.GIF", Color.white, io);
		
		clickRect = new RainbowIntRect(centerX - spawnSixImage.getDesiredWidth()/2, centerY- spawnSixImage.getDesiredHeight() / 2, spawnSixImage.getDesiredWidth(), spawnSixImage.getDesiredHeight(), this);
		
		
	}
	
	
	public int getType() {
		return type;
	}
	
	public void setType(int newT) {
		type = newT;
	}
	
	@Override
	public void draw(Graphics g) {
		
		switch(type) {
		case RainbowElementalSprite.NO_SPAWNING:
			noSpawnImage.drawObjectFromUpperLeftCorner(g, clickRect.getMinX(), clickRect.getMinY());
			break;
		case RainbowElementalSprite.SPAWN_TWO:
			spawnTwoImage.drawObjectFromUpperLeftCorner(g, clickRect.getMinX(), clickRect.getMinY());
			break;
		case RainbowElementalSprite.SPAWN_SIX:
			spawnSixImage.drawObjectFromUpperLeftCorner(g, clickRect.getMinX(), clickRect.getMinY());
			break;
		}
		
		cloudImage.drawObjectFromUpperLeftCorner(g, clickRect.getMinX() + 80, clickRect.getMinY() + 80);
		
		if(flagSpawnPoint != null) {
			flagOutlineImage.drawObjectIgnoreBounds(g, flagSpawnPoint.x, flagSpawnPoint.y);
		}
		
		if(isSelected()) {
			super.drawSelectionOverObject(g);
		}
		
	}

	@Override
	public IntRect getClickRect() {
		return clickRect;
	}

	@Override
	public void setSelected(boolean is) {
		isSelected = is;
	}

	@Override
	public boolean isSelected() {
		return isSelected;
	}


	@Override
	public void setClickRect(IntRect newRect) {
		// TODO Auto-generated method stub
		
	}
	
	
	public void updateMousePosition(int mouseX, int mouseY) {
		
		
		if(mouseY > this.clickRect.getMaxY()) {
			
			//beneath the rainbow
			
			int middleX = this.clickRect.getCenter().x;
			int minX = middleX - ADJUSTED_RADIUS;
			int maxX = middleX + ADJUSTED_RADIUS;
			
			if(mouseX > maxX) {
				this.flagSpawnPoint = new Point(maxX, clickRect.getMaxY());
			} else if(mouseX < minX) {
				this.flagSpawnPoint = new Point(minX, clickRect.getMaxY());
			} else {
				this.flagSpawnPoint = new Point(mouseX, clickRect.getMaxY());
			}
			
		} else {
			
			//within the rainbow, or beyond it?
			
			int xOrigin = (clickRect.getMinX() + clickRect.getMaxX()) / 2;
			int yOrigin = clickRect.getMaxY();
			
			int xDelta = xOrigin - mouseX;
			int yDelta = yOrigin - mouseY;
			
			if(Math.sqrt(xDelta * xDelta + yDelta * yDelta) > ADJUSTED_RADIUS) {
				//outside the rainbow!
				
				if(xDelta == 0) {
					//don't want a divide-by-zero exception!
					flagSpawnPoint = new Point(xOrigin, yOrigin - ADJUSTED_RADIUS);
				} else {
					
					double ratio = Math.abs(((double) yDelta) / ((double) xDelta));
					
//					System.out.println("ratio is: " + ratio);
					
					double relativeFlagXPos = Math.sqrt(ADJUSTED_RADIUS * ADJUSTED_RADIUS / ((1 + ratio*ratio)));
					if(xDelta > 0)
						relativeFlagXPos = -relativeFlagXPos;
					
					double relativeFlagYPos = Math.abs(ratio * relativeFlagXPos);
					relativeFlagYPos = -relativeFlagYPos;
					
					int absoluteFlagXPos = xOrigin + (int) relativeFlagXPos;
					int absoluteFlagYPos = yOrigin + (int) relativeFlagYPos;
					
					flagSpawnPoint = new Point(absoluteFlagXPos, absoluteFlagYPos);
					
					
				}
				
				
			} else {
				//within the rainbow
				
				this.flagSpawnPoint = new Point(mouseX, mouseY);
			}
			
			
		}
		
		
	}
	
	
	public RainbowBossScreenPaletteObject getPaletteObject() {
		return paletteObject;
	}
	

	
	public class RainbowIntRect extends IntRect {
		
		private RainbowBossScreenObject parent;
		
		public RainbowIntRect(int x, int y, int width, int height, RainbowBossScreenObject theParent) {
			super(x,y, width, height);
			
			parent = theParent;
		}
		
		public void move(int xMove, int yMove) {
			super.move(xMove, yMove);
			parent.flagSpawnPoint.x += xMove;
			parent.flagSpawnPoint.y += yMove;
		}
		
	}
	
	
	
	public void writeOutToFile(BufferedWriter out) throws Exception{
		
		out.write("boss:");
		out.newLine();
		
		Point centerPoint = clickRect.getCenter();
		
		out.write("x:");
		out.write(Integer.toString(centerPoint.x));
		out.newLine();
		
		out.write("y:");
		out.write(Integer.toString(clickRect.getMaxY()));
		out.newLine();
		
		if(this.type == RainbowElementalSprite.SPAWN_TWO) {
			out.write("spawnEnemies");
			out.newLine();
		} else if(this.type == RainbowElementalSprite.SPAWN_SIX) {
			out.write("spawnSixEnemies");
			out.newLine();
		}
		
		out.write("safeX:");
		out.write(Integer.toString(this.flagSpawnPoint.x));
		out.newLine();
		
		out.write("safeY:");
		out.write(Integer.toString(this.flagSpawnPoint.y));
		out.newLine();
		
		out.write("/boss");
		out.newLine();
		
		out.newLine();
	}


	public Point getFlagSpawnPoint() {
		return flagSpawnPoint;
	}


	public void setFlagSpawnPoint(Point flagSpawnPoint) {
		this.flagSpawnPoint = flagSpawnPoint;
	}
	
	
	@Override
	public int getLayer() {
		
		return 9;
	}
	
	
}
